/* $Id: semascope.c,v 1.34 1998/09/16 19:32:01 ericb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Keith Bayern, modified by Eric Backus */

/* Scope-like demo program for E1432/E1433

   This program runs a simple auto-arm, auto-trigger time-domain
   measurement.  This makes a nice "warm-fuzzy" test that the E143x
   software is installed properly and your E143x hardware is working.

   By default, the program will search the VXI system for all
   available E143x modules.  If necessary, it will download the
   "sema.bin" firmware into the modules, and will then run a simple
   measurement.  The data from the measurement will get plotted to X11
   windows, in real time.

   Use "./semascope -u" for info on command-line options.  The default
   is to use all modules, a blocksize of 1024, an input range of 1
   Volt, a span of 20 kHz, block mode, and plot data for all channels.
   These defaults can all be changed on the command-line. */

#include <stdio.h>		/* For printf */
#include <stdlib.h>		/* For exit */
#include <string.h>		/* For strrchr */
#include <unistd.h>		/* For getopt */
#include <X11/Xlib.h>		/* For XOpenDisplay */

#include <sicl.h>		/* For iopen */

#include "xplot.h"		/* For xplot_init_plot */
#include "e1432.h"		/* For e1432_xxx */

/* Hard-coded maximums */
#define	NMODLIST_MAX	64	/* Max E143x modules */
#define	NCHAN_MAX	(NMODLIST_MAX * E1432_INPUT_CHANS)

/* Pixel spacing between windows */
#define	X_SPACING	15
#define	Y_SPACING	40

/* Default values for several parameters */
#define	BLOCKSIZE_DEF	1024
#define	RANGE_DEF	1.0
#define	SPAN_DEF	20000.0

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n",\
		       progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n",\
		       progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: semascope.c,v 1.34 1998/09/16 19:32:01 ericb Exp $";
static const char *progname;

static int
detect_modules(int maxlist, SHORTSIZ16 *modlist, int *nmodlist)
{
    INST    sicl_id;		/* Sicl id for the vxi bus */
    struct vxiinfo info;	/* Info about servants */
    int     lalist[255];	/* list of all cards */
    int     i, status, numfound;

    /* Open sicl interface */
    sicl_id = iopen("vxi");
    if (sicl_id == 0)
    {
	(void) fprintf(stderr, "%s: iopen: failed\n", progname);
	return -1;
    }

    /* Fill servant list, -1 for empty entries */
    status = ivxiservants(sicl_id, 255, lalist);
    if (status != 0)
    {
	(void) fprintf(stderr,
		       "%s: ivxiservants: returned %d\n",
		       progname, status);
	(void) iclose(sicl_id);
	return -1;
    }

    /* Check each board, copy any E143x to modlist */
    numfound = 0;
    for (i = 0; lalist[i] >= 0; i++)
    {
	status = ivxirminfo(sicl_id, lalist[i], &info);
	if (status != 0)
	{
	    (void) fprintf(stderr, "%s: ivxirminfo: "
			   "returned %d for laddr %d\n",
			   progname, status, lalist[i]);
	    continue;
	}

	if (info.man_id == E1432_MAN_ID &&
	    (unsigned long) info.model >= E1432_MODEL_CODE_MIN &&
	    (unsigned long) info.model <= E1432_MODEL_CODE_MAX &&
	    numfound < maxlist)
	{
	    if (info.selftest == 0)
	    {
		(void) fprintf(stderr, "%s: E143x FAILED "
			       "self test at laddr %d\n",
			       progname, lalist[i]);
		continue;
	    }
	    modlist[numfound++] = lalist[i];
	}
    }
    *nmodlist = numfound;

    (void) iclose(sicl_id);
    return 0;
}

/* Initialize E143x library, decide which modules to use, download
   code if needed, create channel group. */
static int
init(int nladdr, int laddr[], int *nmodlist, SHORTSIZ16 modlist[],
     int nmodlist_user, int *nchan, SHORTSIZ16 chanlist[],
     E1432ID *hw, int *group, int flipflag)
{
    struct e1432_hwconfig hwconfig[NMODLIST_MAX];
    SHORTSIZ16 modlist_tmp[NMODLIST_MAX];
    int     i, j, status, chan, nmodlist_tmp;

    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(detect_modules(NMODLIST_MAX, modlist, nmodlist));

    /* --- Decide which modules to use --- */

    if (nladdr != 0)
    {
	/* Use requested logical addresses */
	nmodlist_tmp = 0;
	for (j = 0; j < nladdr; j++)
	    for (i = 0; i < *nmodlist; i++)
		if (modlist[i] == laddr[j])
		    modlist_tmp[nmodlist_tmp++] = laddr[j];
	*nmodlist = nmodlist_tmp;
    }
    else
	/* No requested logical addresses, so use all available */
	for (i = 0; i < *nmodlist; i++)
	    modlist_tmp[i] = modlist[i];

    if (*nmodlist == 0)
    {
	if (nladdr == 0)
	    (void) fprintf(stderr, 
			   "%s: no E143x modules found\n",
			   progname);
	else
	    (void) fprintf(stderr,
			   "%s: no E143x modules match requested LA\n",
			   progname);
	return -1;
    }

    (void) printf("Found %d E143x module%s at:",
		  *nmodlist, *nmodlist > 1 ? "s" : "");
    for (i = 0; i < *nmodlist; i++)
    {
	/* The E1432 library makes the last module in the list of
	   modules the system master.  In order to make the first
	   module the system master, we normally reverse the order of
	   the list.  But if flipflag is specified, then we don't
	   reverse the order. */
	if (flipflag)
	    modlist[i] = modlist_tmp[i];
	else
	    modlist[i] = modlist_tmp[*nmodlist - 1 - i];
	(void) printf(" %d", modlist[i]);
    }
    (void) printf("\n");

    if (nmodlist_user > 0 && nmodlist_user < *nmodlist)
    {
	/* Use number of modules specified by user */
	if (nmodlist_user > 1)
	    (void) printf("First %d modules being used\n", nmodlist_user);
	else
	    (void) printf("First module being used\n");
	*nmodlist = nmodlist_user;
    }

    /* --- Finally done deciding which modules to use --- */

#ifdef	STANDALONE
    /* Special code for executable incorporating sema.bin.  In this
       case, don't bother doing e1432_get_hwconfig first, just always
       download sema.bin.  The whole point of the standalone program
       is it works anywhere, and we wouldn't want it to fail just
       because a different sema.bin is in the module. */
    {
	static int sema_bin[] =
	{
#	    include "sema.asc"
	};
	static struct e1432_install_from_mem sema_bin_wrap =
	{
	    sizeof sema_bin, (char *) &sema_bin
	};
	(void) printf("Installing sema.bin in all E143x modules ... ");
	(void) fflush(stdout);
	CHECK(e1432_install(*nmodlist, modlist,
			    E1432_INSTALL_FROM_MEM, &sema_bin_wrap));
	(void) printf("done\n");
	CHECK(e1432_get_hwconfig(*nmodlist, modlist, hwconfig));
    }
#else
    /* Normal case, check to see if we need to download sema.bin. */
    CHECK(e1432_print_errors(0));
    status = e1432_get_hwconfig(*nmodlist, modlist, hwconfig);
    CHECK(e1432_print_errors(1));
    if (status != 0)
    {
	(void) printf("Installing sema.bin from "
		      "/opt/e1432/lib/sema.bin in all E143x modules ... ");
	(void) fflush(stdout);
	CHECK(e1432_install(*nmodlist, modlist, 0,
			    "/opt/e1432/lib/sema.bin"));
	(void) printf("done\n");
	CHECK(e1432_get_hwconfig(*nmodlist, modlist, hwconfig));
    }
#endif

    CHECK(e1432_assign_channel_numbers(*nmodlist, modlist, hw));

    *nchan = 0;
    for (i = 0; i < *nmodlist; i++)
    {
	*nchan += hwconfig[i].input_chans;
	switch (hwconfig[i].model_code)
	{
	case E1432_MODEL_CODE_E1432:
	    (void) printf("E1432 ");
	    break;
	case E1432_MODEL_CODE_E1433:
	    (void) printf("E1433 ");
	    break;
	case E1432_MODEL_CODE_E1434:
	    (void) printf("E1434 ");
	    break;
	default:
	    (void) printf("E143x ");
	    break;
	}
	(void) printf("at LA %3d has %2d input channels\n",
		      modlist[i], hwconfig[i].input_chans);
    }
    if (*nmodlist > 1)
	(void) printf("Total %d input channels\n", *nchan);
    if (*nchan == 0)
    {
	(void) fprintf(stderr, "%s: no input channels found\n",
		       progname);
	return -1;
    }

    for (chan = 0; chan < *nchan; chan++)
	chanlist[chan] = E1432_INPUT_CHAN(chan + 1);

    *group = e1432_create_channel_group(hw, *nchan, chanlist);
    if (*group >= 0)
    {
	(void) fprintf(stderr, "%s: e1432_create_channel_group: "
		       "returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int contflag, int autozeroflag,
      long *blocksize, double range, double *span)
{
    LONGSIZ32 tmp_blocksize;
    FLOATSIZ32 tmp_range, tmp_span;

    CHECK(e1432_set_analog_input(hw, group, E1432_INPUT_MODE_VOLT,
				 E1432_INPUT_SOURCE_BNC,
				 E1432_ANTI_ALIAS_ANALOG_ON,
				 E1432_COUPLING_DC, range));
    CHECK(e1432_set_auto_arm(hw, group, E1432_MANUAL_ARM));
    CHECK(e1432_set_auto_trigger(hw, group, E1432_MANUAL_TRIGGER));
    CHECK(e1432_set_data_format(hw, group, *blocksize,
				E1432_DATA_SIZE_16,
				contflag ? E1432_CONTINUOUS_MODE :
				E1432_BLOCK_MODE,
				E1432_APPEND_STATUS_OFF));
    CHECK(e1432_set_span(hw, group, *span));
    CHECK(e1432_set_trigger(hw, group, E1432_CHANNEL_OFF,
			    0.0, 0.0, 0.0, E1432_TRIGGER_SLOPE_POS,
			    E1432_TRIGGER_MODE_LEVEL));

    /* Print setup */
    CHECK(e1432_get_blocksize(hw, E1432_INPUT_CHAN(1), &tmp_blocksize));
    CHECK(e1432_get_range(hw, E1432_INPUT_CHAN(1), &tmp_range));
    CHECK(e1432_get_span(hw, E1432_INPUT_CHAN(1), &tmp_span));
    *blocksize = tmp_blocksize;
    *span = tmp_span;
    (void) printf("Blocksize:\t%ld\n", tmp_blocksize);
    (void) printf("Range:\t\t%g Volts\n", tmp_range);
    (void) printf("Span:\t\t%g Hz\n", tmp_span);
    (void) printf("Data Mode:\t%s\n",
		  contflag ? "Continuous" : "Block");

    if (autozeroflag)
    {
	(void) printf("Doing autozero ... ");
	(void) fflush(stdout);
	CHECK(e1432_auto_zero(hw, group));
	(void) printf("done\n");
    }

    return 0;
}

static int
allocate(long blocksize, int nchan, FLOATSIZ32 **dataBuffer)
{
    /* Use calloc, so that the buffer holds zeros before we get real
       data, to keep the xplot code happy before the first data
       block. */
    *dataBuffer = calloc(sizeof **dataBuffer, blocksize * nchan);
    if (*dataBuffer == NULL)
    {
	(void) fprintf(stderr,
		       "%s: unable to allocate buffer",
		       progname);
	return -1;
    }

    return 0;
}

/* Open plot windows */
static int
open_windows(int nchan_plot, SHORTSIZ16 chanlist[], char *plotID[],
	     FLOATSIZ32 *dataBuffer, double span,
	     long blocksize, double yscale)
{
    Display *disp;
    char    geometry[32], title[8];
    int     divisor, chan;
    int     xscreen, yscreen, xwidth, ywidth, xstart, ystart;

    /* Get screen size in pixels */
    xscreen = 0;
    yscreen = 0;
    disp = XOpenDisplay("");
    if (disp != 0)
    {
	xscreen = XDisplayWidth(disp, 0);
	yscreen = XDisplayHeight(disp, 0);
	(void) XCloseDisplay(disp);
    }
    if (xscreen == 0 || yscreen == 0)
    {
	/* If anything failed, guess at a size */
	xscreen = 1024;
	yscreen = 768;
    }

    divisor = 2;	/* Maximum window size 1/2 screen dimension */
    while ((divisor * divisor) < nchan_plot)
	divisor++;

    /* Window size in pixels */
    xwidth = xscreen / divisor - X_SPACING;
    ywidth = yscreen / divisor - Y_SPACING;

    for (chan = 0; chan < nchan_plot; chan++)
    {
	/* Window location in pixels */
	xstart = chan % divisor * (xwidth + X_SPACING) + 2;
	ystart = chan / divisor * (ywidth + Y_SPACING) + 3;

	(void) sprintf(geometry, "%dx%d+%d+%d",
		       xwidth, ywidth, xstart, ystart);
	(void) sprintf(title, "%3d", chanlist[chan]);

	plotID[chan] = xplot_init_plot(dataBuffer + chan * blocksize,
				       blocksize, span,
				       yscale, -yscale, TIME_TRACE,
				       geometry, title);
    }

    return 0;
}

static int
run(E1432ID hw, int group, int contflag,
    long blocksize, int nchan, int nchan_plot,
    FLOATSIZ32 *dataBuffer, char *plotID[])
{
    LONGSIZ32 actualsize;
    int     status, chan;

    CHECK(e1432_init_measure(hw, group));

    while (1)
    {
	/* Arm and trigger - first time only for continuous mode,
	   every time for block mode */
	if (contflag < 2)
	{
	    CHECK(e1432_arm_measure(hw, group, 1));
	    CHECK(e1432_trigger_measure(hw, group, 1));
	}
	if (contflag == 1)
	    contflag = 2;

	/* Wait for data */
	while ((status = e1432_block_available(hw, group)) == 0)
	    /* While no data, check for X events */
	    for (chan = 0; chan < nchan_plot; chan++)
		xplot_check_events(plotID[chan]);
	if (status < 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_block_available: returned %d\n",
			   progname, status);
	    return -1;
	}

	/* Read the data */
	CHECK(e1432_read_float32_data(hw, group, E1432_TIME_DATA,
				      dataBuffer, blocksize * nchan,
				      NULL, &actualsize));
	if (actualsize != blocksize * nchan)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_read_float32_data: "
			   "actual count was %ld, expected %ld\n",
			   progname, actualsize, blocksize * nchan);
	    return -1;
	}

	/* Plot the data */
	for (chan = 0; chan < nchan_plot; chan++)
	{
	    xplot_data_update(plotID[chan]);
	    xplot_check_events(plotID[chan]);
	}
    }
    /*NOTREACHED*/
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "E1432/E1433 multi-channel time display for X11\n"
		   "Usage: %s [-cfuVz] [-b blocksize] [-l la] [-n nmodlist]\n"
		   "\t[-N nchan] [-r range] [-s span] [-y yscale]\n"
		   "\t-b: Set blocksize to <blocksize>, default 1024\n"
		   "\t-c: Use continuous mode, default is block mode\n"
		   "\t-f: Reverse (flip) the LA order\n"
		   "\t    (used for some multi-mainframe systems)\n"
		   "\t-l: Use logical address <la>, multiple -l accumulate\n"
		   "\t-n: Use the first <nmodlist> modules, default all\n"
		   "\t-N: Plot the first <nchan> channels, default 16, -1 means all\n"
		   "\t-r: Set range to <range>, default 1 Volt\n"
		   "\t-s: Set span to <span>, default 20 kHz\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n"
		   "\t-y: Set yscale to <yscale>\n"
		   "\t-z: Do autozero\n",
		   progname);
    exit(2);
}

int
main(int argc, char *argv[])
{
    E1432ID hw;
    SHORTSIZ16 modlist[NMODLIST_MAX];
    SHORTSIZ16 chanlist[NCHAN_MAX];
    FLOATSIZ32 *dataBuffer;
    double  range, span, yscale;
    char   *plotID[NCHAN_MAX];
    char   *p;
    long    blocksize;
    int     laddr[NMODLIST_MAX];
    int     c, contflag, flipflag, autozeroflag;
    int     nladdr, nmodlist, nmodlist_user, group, nchan, nchan_plot;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    blocksize = BLOCKSIZE_DEF;
    contflag = 0;
    flipflag = 0;
    nladdr = 0;
    nmodlist_user = 0;
    nchan_plot = 16;	/* Only plot first 16 channels by default */
    range = RANGE_DEF;
    span = SPAN_DEF;
    yscale = RANGE_DEF;
    autozeroflag = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "b:cfl:n:N:r:s:uVy:z")) != EOF)
	switch (c)
	{
	case 'b':		/* Blocksize */
	    blocksize = strtol(optarg, &p, 0);
	    if (optarg == p || blocksize <= 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid blocksize: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'c':		/* Continuous mode */
	    contflag = 1;
	    break;
	case 'f':		/* Reverse la order */
	    flipflag = 1;
	    break;
	case 'l':		/* Logical address */
	    if (nladdr < NMODLIST_MAX)
	    {
		laddr[nladdr] = (SHORTSIZ16) strtol(optarg, &p, 0);
		if (optarg == p || laddr[nladdr] <= 0 ||
		    laddr[nladdr] > 255)
		{
		    (void) fprintf(stderr,
				   "%s: invalid logical address: '%s'\n",
				   progname, optarg);
		    usage();
		}
		nladdr++;
	    }
	    else
	    {
		(void) fprintf(stderr,
			       "%s: too many logical addresses requested\n",
			       progname);
		usage();
	    }
	    break;
	case 'n':		/* Module count */
	    nmodlist_user = strtol(optarg, &p, 0);
	    if (optarg == p || nmodlist_user <= 0 ||
		nmodlist_user > NMODLIST_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid module count: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':		/* Channels to plot */
	    nchan_plot = strtol(optarg, &p, 0);
	    if (optarg == p || nchan_plot > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid plot channel count: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'r':		/* Input range */
	    if (sscanf(optarg, "%lf", &range) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid range: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    yscale = range;
	    break;
	case 's':		/* Span */
	    if (sscanf(optarg, "%lf", &span) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid span: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'y':		/* Yscale */
	    if (sscanf(optarg, "%lf", &yscale) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid yscale: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'z':		/* Autozero */
	    autozeroflag = 1;
	    break;
	case 'u':		/* Usage */
	default:
	    usage();
	}

    /* Initialize the library */
    if (init(nladdr, laddr, &nmodlist, modlist, nmodlist_user, &nchan,
	     chanlist, &hw, &group, flipflag) < 0)
	return EXIT_FAILURE;

    /* Set up the hardware */
    if (setup(hw, group, contflag, autozeroflag,
	      &blocksize, range, &span) < 0)
	return EXIT_FAILURE;

    /* Allocate buffers */
    if (allocate(blocksize, nchan, &dataBuffer) < 0)
	return EXIT_FAILURE;

    /* Open plot windows */
    if (nchan_plot < 0 || nchan_plot > nchan)
	nchan_plot = nchan;
    if (open_windows(nchan_plot, chanlist, plotID, dataBuffer,
		     span, blocksize, yscale) < 0)
	return EXIT_FAILURE;

    /* Run the measurement */
    if (run(hw, group, contflag, blocksize,
	    nchan, nchan_plot, dataBuffer, plotID) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
